#include <time.h>
#include "../common/vector.h"
#include "../common/matrix.h"
#include "../common/misc.h"
#include "../simulation/simulation.h"
#include "solvers.h"
#include "../render/intersect/sphere.h"
#include "../common/debug.h"
#include "../common/colors.h"
#include "../parse/parser.h"
#include "../common/list.h"

#include "collision.h"

extern scene_data *main_scene;
extern int running;
extern state *current_state;

double tessellation_degree;


//create a PQP matrix from a double matrix
void fill_pqp_matrix(PQP_REAL pqp[3][3], double rot[3][3])
{
	pqp[0][0] = rot[0][0];
	pqp[0][1] = rot[0][1];
	pqp[0][2] = rot[0][2];
	
	pqp[1][0] = rot[1][0];
	pqp[1][1] = rot[1][1];
	pqp[1][2] = rot[1][2];
	
	pqp[2][0] = rot[2][0];
	pqp[2][1] = rot[2][1];
	pqp[2][2] = rot[2][2];
}




//constructs a PQP model from the objects tessellation
void add_pqp_data(object *obj)
{
	int j;
	PQP_REAL p1[3], p2[3], p3[3];	
	
	obj->pqp_data = new PQP_Model();
	obj->pqp_data->BeginModel();
	
	for(j=0; j<obj->polygon->num_triangles; j++)
	{
		p1[0] = obj->polygon->triangles[j].vertices[0]->x;
		p1[1] = obj->polygon->triangles[j].vertices[0]->y;
		p1[2] = obj->polygon->triangles[j].vertices[0]->z;
		
		p2[0] = obj->polygon->triangles[j].vertices[1]->x;
		p2[1] = obj->polygon->triangles[j].vertices[1]->y;
		p2[2] = obj->polygon->triangles[j].vertices[1]->z;
		
		p3[0] = obj->polygon->triangles[j].vertices[2]->x;
		p3[1] = obj->polygon->triangles[j].vertices[2]->y;
		p3[2] = obj->polygon->triangles[j].vertices[2]->z;
		
		obj->pqp_data->AddTri(p1, p2, p3, j);
	}
	
	obj->pqp_data->EndModel();
}



//plane collision resolution
void resolve_plane_collision(object *obj1, object *obj2, PQP_CollideResult *collision)
{
	vector plane_normal;
	double sphere_speed;
	
	//get the plane's normal
	obj2->normal(&plane_normal, obj2, &plane_normal);
	
	sphere_speed = vector_length(&obj1->st.velocity);
	
	normalize(&plane_normal);
	normalize(&obj1->st.velocity);
	
	//reflect sphere's velocity
	reflect_vector(&obj1->st.velocity, &plane_normal, &obj1->st.velocity);
	multiply_vector(&obj1->st.velocity, &obj1->st.velocity, sphere_speed);
	//invert_vector(&obj1->st.velocity);
	
	//reverse rotation
	obj1->st.rot_x = -obj1->st.rot_x;
	obj1->st.rot_y = -obj1->st.rot_y;
	obj1->st.rot_z = -obj1->st.rot_z;
	create_rotation_matrix(obj1->rotation, obj1->st.rot_x, obj1->st.rot_y, obj1->st.rot_z);
	
	obj1->collided = 1;
}




//sphere collision resolution
void resolve_sphere_collision(object *obj1, object *obj2, PQP_CollideResult *collision)
{
	vector normal;
	double speed1;
	
	//find normal by getting difference in position
	sub_vectors(&normal, &obj1->pos, &obj2->pos);
	
	speed1 = vector_length(&obj1->st.velocity);
	
	normalize(&normal);
	normalize(&obj1->st.velocity);
	
	//reflect shere1's velocity
	reflect_vector(&obj1->st.velocity, &normal, &obj1->st.velocity);
	multiply_vector(&obj1->st.velocity, &obj1->st.velocity, speed1);
	//invert_vector(&obj1->st.velocity);
	
	//reverse rotation
	obj1->st.rot_x = -obj1->st.rot_x;
	obj1->st.rot_y = -obj1->st.rot_y;
	obj1->st.rot_z = -obj1->st.rot_z;
	create_rotation_matrix(obj1->rotation, obj1->st.rot_x, obj1->st.rot_y, obj1->st.rot_z);
	
	obj1->collided = 1;
}


//sphere collision resolution
void bounce_spheres(object *obj1, object *obj2, PQP_CollideResult *collision)
{
	vector normal;
	vector temp;
	double speed1;
	
	//swap velocities
	copy_vector(&obj1->st.velocity, &temp);
	copy_vector(&obj2->st.velocity, &obj1->st.velocity);
	copy_vector(&temp, &obj2->st.velocity);
	
	obj1->collided = 1;
	obj2->collided = 1;
}


/*
 void resolve_collision(object *obj1, object *obj2, PQP_CollideResult *collision)
 {
	 vector normal1;
	 vector normal2;
	 double speed1;
	 double speed2;
	 
	 //find the point in the middle of the triangle
	 add_vectors(&normal1, obj1->polygon->triangles[collision->Id1(0)].vertices[0], 
				 obj1->polygon->triangles[collision->Id1(0)].vertices[1]);
	 add_vectors(&normal1, &normal1, obj1->polygon->triangles[collision->Id1(0)].vertices[2]);
	 multiply_vector(&normal1, &normal1, 0.3333333);
	 
	 //get its normal
	 obj1->normal(&normal1, obj1, &normal1);
	 
	 //find the point in the middle of the triangle
	 add_vectors(&normal2, obj2->polygon->triangles[collision->Id2(0)].vertices[0], 
				 obj2->polygon->triangles[collision->Id2(0)].vertices[1]);
	 add_vectors(&normal2, &normal2, obj2->polygon->triangles[collision->Id2(0)].vertices[2]);
	 multiply_vector(&normal2, &normal2, 0.3333333);
	 
	 //get its normal
	 obj2->normal(&normal2, obj2, &normal2);
	 
	 speed1 = vector_length(&obj1->st.velocity);
	 speed2 = vector_length(&obj2->st.velocity);
	 
	 normalize(&normal1);
	 normalize(&normal2);
	 normalize(&obj1->st.velocity);
	 //normalize(&obj2->st.velocity);
	 
	 //reflect one object's velocity
	 reflect_vector(&obj1->st.velocity, &normal2, &obj1->st.velocity);
	 
	 multiply_vector(&obj1->st.velocity, &obj1->st.velocity, speed1);
 }
 */



void collide_objects(object *obj1, object *obj2)
{
	int i, j;
	static PQP_REAL R1[3][3], R2[3][3], T1[3], T2[3];
	static PQP_CollideResult collision;
	
	T1[0] = obj1->pos.x;
	T1[1] = obj1->pos.y;
	T1[2] = obj1->pos.z;
	
	T2[0] = obj2->pos.x;
	T2[1] = obj2->pos.y;
	T2[2] = obj2->pos.z;
	
	R1[0][0] = R1[1][1] = R1[2][2] = 1.0;
	R1[0][1] = R1[1][0] = R1[2][0] = 0.0;
	R1[0][2] = R1[1][2] = R1[2][1] = 0.0;
	
	R2[0][0] = R2[1][1] = R2[2][2] = 1.0;
	R2[0][1] = R2[1][0] = R2[2][0] = 0.0;
	R2[0][2] = R2[1][2] = R2[2][1] = 0.0;
	
	//fill_pqp_matrix(R1, obj1->rotation);
	//fill_pqp_matrix(R2, obj2->rotation);
	
	PQP_Collide(&collision, R1, T1, obj1->pqp_data , R2, T2, obj2->pqp_data, PQP_FIRST_CONTACT);
	
	//debug code
	//printf("obj[%i] and obj[%i]\n", obj_num, j);
	//printf("Num BV tests: %d\n", collision.NumBVTests());
	//printf("Num Tri tests: %d\n", collision.NumTriTests());
	//printf("Num contact pairs: %d\n\n", collision.NumPairs());
	
	//resolve collision if needed
	if(collision.NumPairs() > 0)
	{
		//printd(NORMAL, "collision of obj[%i] and obj[%i]\n", obj_num, j);
		
		//resolve_collision(obj1, obj2, &collision);
		if(obj2->obj_type == SPHERE)
			bounce_spheres(obj1, obj2, &collision);
		if(obj2->obj_type == RECTANGLE)
			resolve_plane_collision(obj1, obj2, &collision);
	}
}


void check_collisions(object *obj1, int obj_num)
{
	int i, j;
	object *obj2;
	
	//check collisions
	for(j=0; j<main_scene->num_objects; j++)
	{
		if(main_scene->models[j]->obj_type == SPHERE)
		{
			if(j==obj_num)
				continue; //don't self collide
			
			if(main_scene->models[j]->collided)
				continue;
			
			obj2 = main_scene->models[j];
			
			collide_objects(obj1, obj2);
		}
	}
}


//checks if a possible sphere position collides with a current sphere
char touching_other_sphere(double new_radius, vector *new_position)
{
	int i;
	double radius_distance;
	double position_distance;
	
	for(i=0; i<main_scene->num_objects; i++)
	{
		if(main_scene->models[i]->obj_type == SPHERE)
		{
			radius_distance = new_radius + main_scene->models[i]->radius;
			position_distance = distance_between(new_position, &main_scene->models[i]->pos);
			
			if(radius_distance * 1.2 > position_distance)
				return 1;  //currently colliding
		}
	}
	
	return 0;  //new position is a good position
}



//increases or decreases the tessellation of the spheres
void adjust_tessellation(int delta)
{
	object *obj;
	int i;
	
	tessellation_degree += delta;
	
	if(tessellation_degree < 3)
		tessellation_degree = 3;
	
	//if there are no spheres, we don't retessellate anything
	if(main_scene->num_objects < 8)
		return;
	
	//retessellate all the spheres
	for(i=0; i<main_scene->num_objects; i++)
	{
		if(main_scene->models[i]->obj_type == SPHERE)
		{
			obj = main_scene->models[i];
			//printd(NORMAL, "retessellating obj[%i]\n", obj->id);
			
			//remove the old tessellation
			delete obj->pqp_data;
			obj->pqp_data = NULL;
			free(obj->polygon->points);
			free(obj->polygon->triangles);
			
			//add the new data
			obj->polygon = sphere_polygon(obj, (int)tessellation_degree);	
			add_pqp_data(obj);
		}
	}
}
